<?php

namespace App\Http\Controllers;

use App\Models\Post;
use App\Models\Comment;
use App\Models\Like;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class PostController extends Controller
{
    /**
     * Display all posts (homepage) with search + pagination.
     */
    public function index(Request $request)
    {
        $search = $request->query('q');

        $posts = Post::with(['comments', 'likes'])
            ->when($search, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('title', 'like', '%' . $search . '%')
                      ->orWhere('body', 'like', '%' . $search . '%');
                });
            })
            ->latest()
            ->paginate(9)
            ->withQueryString();

        return view('posts.index', compact('posts', 'search'));
    }

    /**
     * Show posts in a specific category (news/story/lifestyle).
     */
    public function category(Request $request, string $category)
    {
        $allowed = ['news', 'story', 'lifestyle'];
        if (! in_array($category, $allowed)) {
            abort(404);
        }

        $search = $request->query('q');

        $posts = Post::with(['comments', 'likes'])
            ->where('category', $category)
            ->when($search, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('title', 'like', '%' . $search . '%')
                      ->orWhere('body', 'like', '%' . $search . '%');
                });
            })
            ->latest()
            ->paginate(9)
            ->withQueryString();

        return view('posts.category', [
            'posts'    => $posts,
            'category' => $category,
            'search'   => $search,
        ]);
    }

    /**
     * Archive – posts by year.
     */
    public function archiveYear(Request $request, int $year)
    {
        $search = $request->query('q');

        $posts = Post::with(['comments', 'likes'])
            ->whereYear('created_at', $year)
            ->when($search, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('title', 'like', '%' . $search . '%')
                      ->orWhere('body', 'like', '%' . $search . '%');
                });
            })
            ->latest()
            ->paginate(9)
            ->withQueryString();

        return view('posts.archive', [
            'posts'  => $posts,
            'year'   => $year,
            'month'  => null,
            'search' => $search,
        ]);
    }

    /**
     * Archive – posts by year + month.
     */
    public function archiveMonth(Request $request, int $year, int $month)
    {
        if ($month < 1 || $month > 12) {
            abort(404);
        }

        $search = $request->query('q');

        $posts = Post::with(['comments', 'likes'])
            ->whereYear('created_at', $year)
            ->whereMonth('created_at', $month)
            ->when($search, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('title', 'like', '%' . $search . '%')
                      ->orWhere('body', 'like', '%' . $search . '%');
                });
            })
            ->latest()
            ->paginate(9)
            ->withQueryString();

        return view('posts.archive', [
            'posts'  => $posts,
            'year'   => $year,
            'month'  => $month,
            'search' => $search,
        ]);
    }

    /**
     * Show the form for creating a new post (admin only).
     */
    public function create()
    {
        return view('posts.create');
    }

    /**
     * Store a newly created post.
     */
    public function store(Request $request)
    {

        $request->validate([
            'title'    => 'required|max:255',
            'body'     => 'required',
            'category' => 'required|in:news,story,lifestyle',
            'image'    => 'nullable|image|mimes:jpg,jpeg,png|max:10240',
            'video'    => 'nullable|mimes:mp4,mov,avi,wmv|max:10240',
        ]);

        $imagePath = $request->file('image')
            ? $request->file('image')->store('images', 'public')
            : null;

        $videoPath = $request->file('video')
            ? $request->file('video')->store('videos', 'public')
            : null;

        Post::create([
            'title'    => $request->title,
            'slug'     => Str::slug($request->title),
            'body'     => $request->body,
            'category' => $request->category,
            'image'    => $imagePath,
            'video'    => $videoPath,
            'views'    => 0, // 👈 start with 0 views
        ]);

        return redirect()
            ->route('posts.index')
            ->with('success', 'Post created successfully!');
    }

    /**
     * Display a single post.
     */
    public function show(Post $post)
    {
        // increment views each time this post is viewed
        $post->increment('views');

        // eager-load comments and likes for this post
        $post->load(['comments', 'likes']);

        return view('posts.show', compact('post'));
    }

    /**
     * Show the form for editing the specified post.
     */
    public function edit(Post $post)
    {
        return view('posts.edit', compact('post'));
    }

    /**
     * Update the specified post in storage.
     */
    public function update(Request $request, Post $post)
    {
        $request->validate([
            'title'    => 'required|max:255',
            'body'     => 'required',
            'category' => 'required|in:news,story,lifestyle',
            'image'    => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
            'video'    => 'nullable|mimes:mp4,mov,avi,wmv|max:10240',
        ]);

        $data = [
            'title'    => $request->title,
            'slug'     => Str::slug($request->title),
            'body'     => $request->body,
            'category' => $request->category,
        ];

        // Handle image update
        if ($request->hasFile('image')) {
            if ($post->image) {
                Storage::disk('public')->delete($post->image);
            }
            $data['image'] = $request->file('image')->store('images', 'public');
        }

        // Handle video update
        if ($request->hasFile('video')) {
            if ($post->video) {
                Storage::disk('public')->delete($post->video);
            }
            $data['video'] = $request->file('video')->store('videos', 'public');
        }

        $post->update($data);

        return redirect()
            ->route('posts.index')
            ->with('success', 'Post updated successfully!');
    }

    /**
     * Remove the specified post from storage.
     */
    public function destroy(Post $post)
    {
        if ($post->image) {
            Storage::disk('public')->delete($post->image);
        }

        if ($post->video) {
            Storage::disk('public')->delete($post->video);
        }

        $post->delete();

        return redirect()
            ->route('posts.index')
            ->with('success', 'Post deleted successfully!');
    }

    /**
     * Store a comment for a post.
     */
    public function storeComment(Request $request, Post $post)
    {
        $request->validate([
            'name'  => 'required|string|max:255',
            'email' => 'nullable|email',
            'body'  => 'required|string',
        ]);

        Comment::create([
            'post_id' => $post->id,
            'name'    => $request->name,
            'email'   => $request->email,
            'body'    => $request->body,
        ]);

        return back()->with('success', 'Comment added successfully!');
    }

    /**
     * Like a post (per IP).
     */
    public function like(Request $request, Post $post)
    {
        $ip = $request->ip();

        Like::firstOrCreate([
            'post_id'    => $post->id,
            'ip_address' => $ip,
        ]);

        return back()->with('success', 'You liked this post!');
    }
}
